#include <common.h>
#include <command.h>
#include <w5300_mac.h>
#include <net.h>

#ifdef CONFIG_DRIVER_W5300

#if (CONFIG_COMMANDS & CFG_CMD_NET)

//#undef DEBUG

#define DEBUG_W5300	1
#if DEBUG_W5300
	#define DPRINTF(args...) printf(args)
#else
	#define DPRINTF(args...) { }
#endif /* DEBUG_W5300 */

static u16 swap16(u16 x)
{
	return ((((x) & 0x00ff) << 8) | ( (x) >> 8));
}

/* packet page register access functions */
static void eth_reset (void)
{
	/* reset NIC */
	IINCHIP_WRITE(MR, MR_RST);

	/* wait for 200ms */
	udelay (200000);
	/* Wait until the chip is reset */

}

void disp_lcd_ip(IPaddr_t ip)
{
	unsigned char buffer[17];
	
	lcd_gotoxy(0,1);
	
	sprintf(buffer, "%03d.%03d.%03d.%03d", (ip>>24)&0xFF, (ip>>16)&0xFF, (ip>>8)&0xFF, ip&0xFF);
	lcd_puts(buffer);	
}

static void eth_reginit (void)
{
	u16 i, ssum=0, rsum=0, mem_cfg=0;
	u8 tx_size[8] = {0};
	u8 rx_size[8] = {0};
	u8 mtx_size[W5300_MAX_CHANNEL] = W5300_TMSR_ARRAY;
	u8 mrx_size[W5300_MAX_CHANNEL] = W5300_RMSR_ARRAY;
	
	for(i=0; i<W5300_MAX_CHANNEL; i++) {
		tx_size[i] = mtx_size[i];
		rx_size[i] = mrx_size[i];
	}

	for(i=0; i<8; i++) {
		if(tx_size[i] > 64) tx_size[i] = 64;
		if(rx_size[i] > 64) rx_size[i] = 64;
		
		ssum += (u16)tx_size[i];
		rsum += (u16)rx_size[i];
	}

	if( ((ssum % 8) != 0) || ((ssum + rsum) != 128) ) {
		// Wrong, applying defaults
		ssum = rsum = 64;
		tx_size[0] = rx_size[0] = tx_size[1] = rx_size[1] = 32;
		tx_size[2] = rx_size[2] = tx_size[3] = rx_size[3] = 0;
		tx_size[4] = rx_size[4] = tx_size[5] = rx_size[5] = 0;
		tx_size[6] = rx_size[6] = tx_size[7] = rx_size[7] = 0;
	}
	
	IINCHIP_WRITE(TMS01R,((u16)tx_size[0] << 8) + (u16)tx_size[1]);
	IINCHIP_WRITE(TMS23R,((u16)tx_size[2] << 8) + (u16)tx_size[3]);
	IINCHIP_WRITE(TMS45R,((u16)tx_size[4] << 8) + (u16)tx_size[5]);
	IINCHIP_WRITE(TMS67R,((u16)tx_size[6] << 8) + (u16)tx_size[7]);

	IINCHIP_WRITE(RMS01R,((u16)rx_size[0] << 8) + (u16)rx_size[1]);
	IINCHIP_WRITE(RMS23R,((u16)rx_size[2] << 8) + (u16)rx_size[3]);
	IINCHIP_WRITE(RMS45R,((u16)rx_size[4] << 8) + (u16)rx_size[5]);
	IINCHIP_WRITE(RMS67R,((u16)rx_size[6] << 8) + (u16)rx_size[7]);

	for(i=0; i<(ssum/8); i++) {
		mem_cfg <<= 1;
		mem_cfg |= 1;
	}

	IINCHIP_WRITE(MTYPER, mem_cfg);

	// channel 0 : MACRAW mode
	i = 0;
MACRAW:
	if(i++ > 10000) {
		DPRINTF("MACRAW open failed...");
		return;
	}
	IINCHIP_WRITE(Sn_MR(0), Sn_MR_MACRAW);
	IINCHIP_WRITE(Sn_CR(0), Sn_CR_OPEN);
	udelay(10000);		// 10ms
	//DPRINTF("Sn_SSR = 0x%04x\n", IINCHIP_READ(Sn_SSR(0)));
	if((IINCHIP_READ(Sn_SSR(0)) & 0xff) != SOCK_MACRAW) {
		IINCHIP_WRITE(Sn_CR(0), Sn_CR_CLOSE);
		goto MACRAW;
	}
}

void eth_halt (void)
{
	/* disable transmitter/receiver mode */
	IINCHIP_WRITE(MR, 0xf900);
	/* "shutdown" to show ChipID or kernel wouldn't find he cs8900 ... */
}

int eth_init (bd_t * bd)
{
	IPaddr_t ip;

	eth_reset();

	/* MR, IMR, ICFGR */
	IINCHIP_WRITE(MR, 0xb900);

	IINCHIP_WRITE(IMR, 0x0);	// no interrupt
	IINCHIP_WRITE(ICFGR, 0x0);

	/* verify chip id */
	if (IINCHIP_READ(IDR) != 0x5300) {
		printf ("W5300 Ethernet chip not found?! IDR=0x%04x\n", IINCHIP_READ(IDR));
		return 0;
	}
	else
	{
		printf ("W5300 Ethernet chip initialized MR=0x%04x\n", IINCHIP_READ(MR));
	}

	/* set the ethernet address */
	IINCHIP_WRITE(SHAR,(((u16)bd->bi_enetaddr[0])<<8)+bd->bi_enetaddr[1]);
	IINCHIP_WRITE(SHAR2,((u16)bd->bi_enetaddr[2]<<8)+bd->bi_enetaddr[3]);
	IINCHIP_WRITE(SHAR4,((u16)bd->bi_enetaddr[4]<<8)+bd->bi_enetaddr[5]);

	/* set the IP,Gateway,Subnet */
	ip = bd->bi_ip_addr;
	ip = ntohl(ip);
	
	//display sip to lcd
	disp_lcd_ip(ip);
	
	
	IINCHIP_WRITE(SIPR, ip>>16);
	IINCHIP_WRITE(SIPR2, ip&0xffff);
	
	ip = getenv_IPaddr ("netmask");
	ip = ntohl(ip);
	
	IINCHIP_WRITE(SUBR, ip>>16);
    IINCHIP_WRITE(SUBR2, ip&0xffff);
	
	ip = getenv_IPaddr ("gatewayip");
	ip = ntohl(ip);
	IINCHIP_WRITE(GAR, ip>>16);
	IINCHIP_WRITE(GAR2, ip&0xffff);

	eth_reginit ();
	return 0;
}

/* Get a data block via Ethernet */
extern int eth_rx (void)
{
	int i;
	unsigned short rxlen, dummy;
	unsigned short *addr;
	u8 *addr8;

	u32 received_rx_size = 0;

	// check recv length
	received_rx_size = IINCHIP_READ(Sn_RX_RSR(0));
	received_rx_size = (received_rx_size << 16) + IINCHIP_READ(Sn_RX_RSR2(0));
	if(received_rx_size == 0) return 0;
	
	addr = (unsigned short *)NetRxPackets[0];
	addr8 = (u8*)NetRxPackets[0];

	dummy = IINCHIP_READ(Sn_RX_FIFOR(0));
	rxlen = (dummy >> 8) | (dummy << 8);
//	DPRINTF("rxlen = %d\n", rxlen);

	for(i=(rxlen >> 1); i>0; i--) {
		*addr++ = IINCHIP_READ(Sn_RX_FIFOR(0));
	}
	if (rxlen & 1) {
		*addr++ = IINCHIP_READ(Sn_RX_FIFOR(0));
	}
//	for(i=rxlen; i>0; i--) DPRINTF("0x%02x ", *addr8++);

	dummy = IINCHIP_READ(Sn_RX_FIFOR(0));
	dummy = IINCHIP_READ(Sn_RX_FIFOR(0));

	IINCHIP_WRITE(Sn_CR(0), Sn_CR_RECV);

	/* Pass the packet up to the protocol layers. */
	NetReceive (NetRxPackets[0], rxlen);

	return rxlen;
}

/* Send a data block via Ethernet. */
extern int eth_send (volatile void *packet, int length)
{
	volatile unsigned short *addr;
	volatile u8 *addr8;
	unsigned short tmp16;
	int tmo;
	unsigned short s;
	u32 free_tx_size = 0;

	free_tx_size = IINCHIP_READ(Sn_TX_FSR(0));
	free_tx_size = (free_tx_size << 16) + IINCHIP_READ(Sn_TX_FSR2(0));
	if(free_tx_size < length) {
#ifdef DEBUG
		printf("Not enough free size.\n");
#endif
		eth_reset();
		eth_reginit();
		return 0;
	}

	if( (length < 0) || (length > 65535) ) DPRINTF("eth_send:exceed length!\n");
	IINCHIP_WRITE(Sn_TX_WRSR(0), (u16)(length >> 16));
	IINCHIP_WRITE(Sn_TX_WRSR2(0), (u16)length);

	/* Write the contents of the packet */
	/* assume even number of bytes */
	addr8 = packet;
	for (addr = packet; length > 0; length -= 2)
	{
		IINCHIP_WRITE(Sn_TX_FIFOR(0), *addr++);
//		DPRINTF("0x%02x ", *addr8++);
//		DPRINTF("0x%02x ", *addr8++);
	}

	IINCHIP_WRITE(Sn_CR(0), Sn_CR_SEND);

	/* wait for transfer to succeed */
	tmo = get_timer (0) + 5 * CFG_HZ;
	while ((s = IINCHIP_READ(Sn_IR(0)) & Sn_IR_SEND_OK) != Sn_IR_SEND_OK) {
		if (get_timer (0) >= tmo)
			return 0;
	}

	IINCHIP_WRITE(Sn_IR(0), Sn_IR_SEND_OK);

	return 0;
}

#endif	/* COMMANDS & CFG_NET */

#endif	/* CONFIG_DRIVER_W5300 */
