/*
 *  armboot - Startup Code for ARM920 CPU-core
 *
 *  Copyright (c) 2001	Marius Grger <mag@sysgo.de>
 *  Copyright (c) 2002	Alex Zpke <azu@sysgo.de>
 *  Copyright (c) 2002	Gary Jennejohn <gj@denx.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */


#include <config.h>
#include <version.h>

#define GPIO_BIT(x)                     ( 1 << x )
#define GPIO_SET(shift,bit)             ((bit) << (shift))

#define _LED_0                          GPIO_BIT(10)
#define _LED_1                          GPIO_BIT(11)
#define S3C2410_REG_GP_BASE             0x56000000
#define S3C2410_REG_OFFSET_GPGCON       0x60
#define S3C2410_REG_OFFSET_GPGDAT       0x64
#define S3C2410_REG_OFFSET_GPGUP        0x68
#define S3C2410_REG_OFFSET_GPHCON       0x70
#define S3C2410_REG_OFFSET_GPHDAT       0x74
#define S3C2410_REG_OFFSET_GPHUP        0x78

#define GPIO_ALT_FN_1                   0x2
#define GPIO_OUT                        0x1

#define GP_RXD2		GPIO_SET(14, GPIO_ALT_FN_1)
#define GP_TXD2		GPIO_SET(12, GPIO_ALT_FN_1)
#define GP_RXD1		GPIO_SET(10, GPIO_ALT_FN_1)
#define GP_TXD1		GPIO_SET( 8, GPIO_ALT_FN_1)
#define GP_RXD0		GPIO_SET( 6, GPIO_ALT_FN_1)
#define GP_TXD0		GPIO_SET( 4, GPIO_ALT_FN_1)

#define GP_LED0         GPIO_SET(20, GPIO_OUT)
#define GP_LED1         GPIO_SET(22, GPIO_OUT)

#define GPIOH_VALUE	( GP_RXD2 | GP_TXD2     \
			| GP_RXD1 | GP_TXD1     \
			| GP_RXD0 | GP_TXD0 )

#define GPIOG_VALUE		(GP_LED0 | GP_LED1)
/*
 *************************************************************************
 *
 * Jump vector table as in table 3.1 in [1]
 *
 *************************************************************************
 */


.globl _start
_start: b       reset
	ldr	pc, _undefined_instruction
	ldr	pc, _software_interrupt
	ldr	pc, _prefetch_abort
	ldr	pc, _data_abort
	ldr	pc, _not_used
	ldr	pc, _irq
	ldr	pc, _fiq

_undefined_instruction:	.word undefined_instruction
_software_interrupt:	.word software_interrupt
_prefetch_abort:	.word prefetch_abort
_data_abort:		.word data_abort
_not_used:		.word not_used
_irq:			.word irq
_fiq:			.word fiq

	.balignl 16,0xdeadbeef


/*
 *************************************************************************
 *
 * Startup Code (reset vector)
 *
 * do important init only if we don't start from memory!
 * relocate armboot to ram
 * setup stack
 * jump to second stage
 *
 *************************************************************************
 */

_TEXT_BASE:
	.word	TEXT_BASE

.globl _armboot_start
_armboot_start:
	.word _start

/*
 * These are defined in the board-specific linker script.
 */
.globl _bss_start
_bss_start:
	.word __bss_start

.globl _bss_end
_bss_end:
	.word _end

#ifdef CONFIG_USE_IRQ
/* IRQ stack memory (calculated at run-time) */
.globl IRQ_STACK_START
IRQ_STACK_START:
	.word	0x0badc0de

/* IRQ stack memory (calculated at run-time) */
.globl FIQ_STACK_START
FIQ_STACK_START:
	.word 0x0badc0de
#endif

#if defined(CONFIG_BOOT_DEVICE_NAND) && defined(CONFIG_S3C2410)
/* some parameters for the board */
NFDATA:         .word   0x4E00000C
NFCMD:          .word   0x4E000004
NFADDR:         .word   0x4E000008

NFCONF:         .word   0x4E000000
NFSTAT:         .word   0x4E000010
NFECC:          .word   0x4E000014

BL_BASE:        .word   TEXT_BASE       /* base address of the bootloader */
NAND_BASE:      .word   0x00000000      /* start address of image in NAND flash */
NAND_TOP:       .word   0x00020000      /* end address of image in NAND flash */
#endif

/*
 * the actual reset code
 */

reset:
	/*
	 * set the cpu to SVC32 mode
	 */
	mrs	r0,cpsr
	bic	r0,r0,#0x1f
	orr	r0,r0,#0xd3
	msr	cpsr,r0

/* turn off the watchdog */
#if defined(CONFIG_S3C2400)
#define pWTCON		0x15300000
/* Interupt-Controller base addresses */
#define INTMSK		0x14400008
/* clock divisor register */
#define CLKDIVN		0x14800014
#elif defined(CONFIG_S3C2410)
#define pWTCON		0x53000000
/* Interupt-Controller base addresses */
#define INTMSK		0x4A000008
#define INTSUBMSK	0x4A00001C
/* About clock */
#define CLKDIVN		0x4C000014

#define LOCKTIME	0x4C000000
#define MPLLCON		0x4C000004
#define UPLLCON		0x4C000008

#define LOCKTIME_VALUE	0x00FFFFFF
#define MPLL_MDIV_266M	(0x7d << 12)
#define MPLL_PDIV_266M	(0x1  << 4)
#define MPLL_SDIV_266M	(0x1  << 0)
#define MPLLCON_VALUE	( MPLL_MDIV_266M |  MPLL_PDIV_266M | MPLL_SDIV_266M )

#define UPLL_MDIV	( 0x78 << 12 )
#define UPLL_PDIV	( 0x2  << 4  )
#define UPLL_SDIV	( 0x3  << 0  )
#define UPLLCON_VALUE	( UPLL_MDIV |  UPLL_PDIV | UPLL_SDIV )
#endif

	ldr     r0, =pWTCON
	mov     r1, #0x0
	str     r1, [r0]

	/*
	 * mask all IRQs by setting all bits in the INTMR - default
	 */
	mov	r1, #0xffffffff
	ldr	r0, =INTMSK
	str	r1, [r0]
#if defined(CONFIG_S3C2410)
	ldr	r1, =0x3ff
	ldr	r0, =INTSUBMSK
	str	r1, [r0]
#endif

	/* FCLK:HCLK:PCLK = 1:2:4 */
	/* default FCLK is 120 MHz ! */
	ldr	r0, =CLKDIVN
	mov	r1, #3
	str	r1, [r0]

	ldr     r0, =S3C2410_REG_GP_BASE

	ldr	r1, =GPIOH_VALUE
	str	r1, [r0, #S3C2410_REG_OFFSET_GPHCON]

	ldr     r1, =GPIOG_VALUE
	str     r1, [r0, #S3C2410_REG_OFFSET_GPGCON]

	// debug led off
	ldr	r0, =S3C2410_REG_GP_BASE
	mov	r1, #(_LED_1)
	str	r1, [r0, #S3C2410_REG_OFFSET_GPGDAT]

	/*
	 * we do sys-critical inits only at reboot,
	 * not when booting from ram!
	 */
#ifdef CONFIG_INIT_CRITICAL
	bl	cpu_init_crit
#endif

relocate:				/* relocate U-Boot to RAM	    */

#if defined(CONFIG_BOOT_DEVICE_NAND) && defined(CONFIG_S3C2410)
	/* Initializng NAND flash controller */
	ldr	r1, NFCONF	/* Set timing */
	and	r0, r0, #0x0
	add	r0, r0, #0x8000 /* (0x1<<15) Enable NAND Flash Controller */
	add	r0, r0, #0x4000 /* (0x2<<13) Reverved */
	add	r0, r0, #0x1000 /* (0x1<<12) Initialize ECC */
	add	r0, r0, #0x800  /* (0x1<<11) NAND Flash nFCE = H (inactive) */
				/* (0x0<<8)  CLE & ALE Duration Settig: TACLS=0 */
				/* (0x0<<7)  Reverved */
	add	r0, r0, #0x20   /* (0x2<<4)  TWRPH0 Duration Setting: TWRPH0=2 */
				/* (0x0<<3)  Reverved */
				/* (0x0<<0)  TWRPH1 Duration Setting: TWRPH0=0 */
	strh	r0, [r1]

	/* Detecting type - 3 or 4 address type */

	/* Chip Enable */
	ldr 	r1, NFCONF
	ldrh	r0, [r1]
	bic	r0, r0, #0x800 /* nFCE = L */
	strh	R0, [R1]

	/* read id command */
	ldr	r1, NFCMD
	mov	r0, #0x90
	strb	r0, [r1]

	/* Set address */
	ldr	r1, NFADDR
	mov	r0, #0x0
	strb	r0, [r1]

wait_ready0:
	ldr	r1, NFSTAT
	ldrh	r0, [r1]
	and	r0, r0, #0x1
	cmp	r0, #0x1
	bne	wait_ready0

	/* read mfr */
	ldr	r1, NFDATA
	ldr	r2, [r1]
	mov	r0, r2
	cmp	r0, #0xec /* Samsung */
	beq	read_id
	cmp	r0, #0x98 /* Tosiba */
	beq	read_id
	b	nand_fail

read_id:
	/* read id */
	ldr	r1, NFDATA
	ldr	r2, [r1]
	mov	r0, r2

	/* nand address step is 3 */
	mov	r7, #0x0

	cmp	r0, #0x76 /* Samsung K9F1208 */
	beq	nand_addr_4
	cmp	r0, #0x79 /* Tosiba TH58100FT */
	beq	nand_addr_4
	b	nand_ok

nand_addr_4:
	/* nand address step is 4 */
	mov	r7, #0x1
	b	nand_ok

nand_fail:
	b	.

nand_ok:

	ldr	r5, BL_BASE	@ R5 contains current RAM address
	ldr	r6, NAND_BASE	@ R6 contains current flash address
	ldr	r9, NAND_TOP	@ R9 contains top of the flash address

/* Read loop */
read_command:
/* Chip Enable */
	ldr 	r1, NFCONF
	ldrh	r0, [r1]
	bic	r0, r0, #0x800 /* nFCE = L */
	strh	R0, [R1]


/*	Set bank to 0 */
/*	Set read command (NAND_CMD_READ) */
	ldr	r1, NFCMD
	mov	r0, #0x0
	strb	r0, [r1]

	@ Set address 1
	ldr	r1, NFADDR
	mov	r0, r6
	strb	r0, [r1]
	@ Set address 2
	mov	r0, r6, lsr #9
	strb	r0, [r1]
	@ Set address 3
	mov	r0, r6, lsr #17
	strb	r0, [r1]
	@ Set address 4 (if needed, according to the type)
	cmp	r7, #0x1
	bne	wait_ready
	mov	r0, r6, lsr #25
	strb	r0, [r1]

/*	Wait ready */
wait_ready:
	ldr	r1, NFSTAT
	ldrh	r0, [r1]
	and	r0, r0, #0x1
	cmp	r0, #0x1
	bne	wait_ready

/*	Get data */
	and	r8, r8, #0x0	@ R8 works as word counter
get_page:
	ldr	r1, NFDATA
	ldr	r2, [r1]
	mov	r0, r2

	ldr	r2, [r1]
	mov	r2, r2, lsl #8
	and	r2, r2, #0x0000ff00
	add	r0, r0, r2

	ldr	r2, [r1]
	mov	r2, r2, lsl #16
	and	r2, r2, #0x00ff0000
	add	r0, r0, r2

	ldr	r2, [r1]
	mov	r2, r2, lsl #24
	and	r2, r2, #0xff000000
	add	r0, r0, r2

	str	r0, [r5]
	add	r8, r8, #0x4
	add	r5, r5, #0x4
	cmp	r8, #0x200
	bne	get_page

	add	r6, r6, #0x200
	cmp	r6, r9
	ble	read_command

/* Chip Disable */
	ldr 	r1, NFCONF
	ldrh	r0, [r1]
	orr	r0, r0, #0x800 /* nFCE = H */
	strh	R0, [R1]

/*	Repeat above cycle according to the number of pages */

	ldr	pc, _real_boot_loader
_real_boot_loader:	.word nand_relocate
nand_relocate:

#else /* defined(CONFIG_BOOT_DEVICE_NAND) && defined(CONFIG_S3C2410) */
	adr	r0, _start		/* r0 <- current position of code   */
	ldr	r1, _TEXT_BASE		/* test if we run from flash or RAM */
	cmp     r0, r1                  /* don't reloc during debug         */
	beq     stack_setup

	ldr	r2, _armboot_start
	ldr	r3, _bss_start
	sub	r2, r3, r2		/* r2 <- size of armboot            */
	add	r2, r0, r2		/* r2 <- source end address         */

copy_loop:
	ldmia	r0!, {r3-r10}		/* copy from source address [r0]    */
	stmia	r1!, {r3-r10}		/* copy to   target address [r1]    */
	cmp	r0, r2			/* until source end addreee [r2]    */
	ble	copy_loop

#endif /* defined(CONFIG_BOOT_DEVICE_NAND) && defined(CONFIG_S3C2410) */

	/* Set up the stack						    */
stack_setup:
	ldr	r0, _TEXT_BASE		/* upper 128 KiB: relocated uboot   */
	sub	r0, r0, #CFG_MALLOC_LEN	/* malloc area                      */
	sub	r0, r0, #CFG_GBL_DATA_SIZE /* bdinfo                        */
#ifdef CONFIG_USE_IRQ
	sub	r0, r0, #(CONFIG_STACKSIZE_IRQ+CONFIG_STACKSIZE_FIQ)
#endif
	sub	sp, r0, #12		/* leave 3 words for abort-stack    */

clear_bss:
	ldr	r0, _bss_start		/* find start of bss segment        */
	ldr	r1, _bss_end		/* stop here                        */
	mov 	r2, #0x00000000		/* clear                            */

clbss_l:str	r2, [r0]		/* clear loop...                    */
	add	r0, r0, #4
	cmp	r0, r1
	bne	clbss_l
	ldr	pc, _start_armboot

_start_armboot:	.word start_armboot


/*
 *************************************************************************
 *
 * CPU_init_critical registers
 *
 * setup important registers
 * setup memory timing
 *
 *************************************************************************
 */


cpu_init_crit:
	/*
	 * flush v4 I/D caches
	 */
	mov	r0, #0
	mcr	p15, 0, r0, c7, c7, 0	/* flush v3/v4 cache */
	mcr	p15, 0, r0, c8, c7, 0	/* flush v4 TLB */

	/*
	 * disable MMU stuff and caches
	 */
	mrc	p15, 0, r0, c1, c0, 0
	bic	r0, r0, #0x00002300	@ clear bits 13, 9:8 (--V- --RS)
	bic	r0, r0, #0x00000087	@ clear bits 7, 2:0 (B--- -CAM)
	orr	r0, r0, #0x00000002	@ set bit 2 (A) Align
	orr	r0, r0, #0x00001000	@ set bit 12 (I) I-Cache
	mcr	p15, 0, r0, c1, c0, 0


	/*
	 * before relocating, we have to setup RAM timing
	 * because memory timing is board-dependend, you will
	 * find a memsetup.S in your board directory.
	 */
	mov	ip, lr
	bl	memsetup
	mov	lr, ip

	mov	pc, lr


/*
 *************************************************************************
 *
 * Interrupt handling
 *
 *************************************************************************
 */

@
@ IRQ stack frame.
@
#define S_FRAME_SIZE	72

#define S_OLD_R0	68
#define S_PSR		64
#define S_PC		60
#define S_LR		56
#define S_SP		52

#define S_IP		48
#define S_FP		44
#define S_R10		40
#define S_R9		36
#define S_R8		32
#define S_R7		28
#define S_R6		24
#define S_R5		20
#define S_R4		16
#define S_R3		12
#define S_R2		8
#define S_R1		4
#define S_R0		0

#define MODE_SVC 0x13
#define I_BIT	 0x80

/*
 * use bad_save_user_regs for abort/prefetch/undef/swi ...
 * use irq_save_user_regs / irq_restore_user_regs for IRQ/FIQ handling
 */

	.macro	bad_save_user_regs
	sub	sp, sp, #S_FRAME_SIZE
	stmia	sp, {r0 - r12}			@ Calling r0-r12
	ldr	r2, _armboot_start
	sub	r2, r2, #(CONFIG_STACKSIZE+CFG_MALLOC_LEN)
	sub	r2, r2, #(CFG_GBL_DATA_SIZE+8)  @ set base 2 words into abort stack
	ldmia	r2, {r2 - r3}			@ get pc, cpsr
	add	r0, sp, #S_FRAME_SIZE		@ restore sp_SVC

	add	r5, sp, #S_SP
	mov	r1, lr
	stmia	r5, {r0 - r3}			@ save sp_SVC, lr_SVC, pc, cpsr
	mov	r0, sp
	.endm

	.macro	irq_save_user_regs
	sub	sp, sp, #S_FRAME_SIZE
	stmia	sp, {r0 - r12}			@ Calling r0-r12
	add     r8, sp, #S_PC
	stmdb   r8, {sp, lr}^                   @ Calling SP, LR
	str     lr, [r8, #0]                    @ Save calling PC
	mrs     r6, spsr
	str     r6, [r8, #4]                    @ Save CPSR
	str     r0, [r8, #8]                    @ Save OLD_R0
	mov	r0, sp
	.endm

	.macro	irq_restore_user_regs
	ldmia	sp, {r0 - lr}^			@ Calling r0 - lr
	mov	r0, r0
	ldr	lr, [sp, #S_PC]			@ Get PC
	add	sp, sp, #S_FRAME_SIZE
	subs	pc, lr, #4			@ return & move spsr_svc into cpsr
	.endm

	.macro get_bad_stack
	ldr	r13, _armboot_start		@ setup our mode stack
	sub	r13, r13, #(CONFIG_STACKSIZE+CFG_MALLOC_LEN)
	sub	r13, r13, #(CFG_GBL_DATA_SIZE+8) @ reserved a couple spots in abort stack

	str	lr, [r13]			@ save caller lr / spsr
	mrs	lr, spsr
	str     lr, [r13, #4]

	mov	r13, #MODE_SVC			@ prepare SVC-Mode
	@ msr	spsr_c, r13
	msr	spsr, r13
	mov	lr, pc
	movs	pc, lr
	.endm

	.macro get_irq_stack			@ setup IRQ stack
	ldr	sp, IRQ_STACK_START
	.endm

	.macro get_fiq_stack			@ setup FIQ stack
	ldr	sp, FIQ_STACK_START
	.endm

/*
 * exception handlers
 */
	.align  5
undefined_instruction:
	get_bad_stack
	bad_save_user_regs
	bl 	do_undefined_instruction

	.align	5
software_interrupt:
	get_bad_stack
	bad_save_user_regs
	bl 	do_software_interrupt

	.align	5
prefetch_abort:
	get_bad_stack
	bad_save_user_regs
	bl 	do_prefetch_abort

	.align	5
data_abort:
	get_bad_stack
	bad_save_user_regs
	bl 	do_data_abort

	.align	5
not_used:
	get_bad_stack
	bad_save_user_regs
	bl 	do_not_used

#ifdef CONFIG_USE_IRQ

	.align	5
irq:
	get_irq_stack
	irq_save_user_regs
	bl 	do_irq
	irq_restore_user_regs

	.align	5
fiq:
	get_fiq_stack
	/* someone ought to write a more effiction fiq_save_user_regs */
	irq_save_user_regs
	bl 	do_fiq
	irq_restore_user_regs

#else

	.align	5
irq:
	get_bad_stack
	bad_save_user_regs
	bl 	do_irq

	.align	5
fiq:
	get_bad_stack
	bad_save_user_regs
	bl 	do_fiq

#endif

	.align	5
.globl reset_cpu
reset_cpu:
#ifdef CONFIG_S3C2400
	bl	disable_interrupts
# ifdef CONFIG_TRAB
	bl	disable_vfd
# endif
	ldr	r1, _rWTCON
	ldr	r2, _rWTCNT
	/* Disable watchdog */
	mov	r3, #0x0000
	str	r3, [r1]
	/* Initialize watchdog timer count register */
	mov	r3, #0x0001
	str	r3, [r2]
	/* Enable watchdog timer; assert reset at timer timeout */
	mov	r3, #0x0021
	str	r3, [r1]
_loop_forever:
	b	_loop_forever
_rWTCON:
	.word	0x15300000
_rWTCNT:
	.word	0x15300008
#else /* ! CONFIG_S3C2400 */
	mov     ip, #0
	mcr     p15, 0, ip, c7, c7, 0           @ invalidate cache
	mcr     p15, 0, ip, c8, c7, 0           @ flush TLB (v4)
	mrc     p15, 0, ip, c1, c0, 0           @ get ctrl register
	bic     ip, ip, #0x000f                 @ ............wcam
	bic     ip, ip, #0x2100                 @ ..v....s........
	mcr     p15, 0, ip, c1, c0, 0           @ ctrl register
	mov     pc, r0
#endif /* CONFIG_S3C2400 */
