/**
 * \file    w5300e01.c
 * W5300 example applications
 * 
 * This file implements the WIZnet W5300E01-ARM EVB Boot configuration
 * 
 * \author wiznet
 * \date 15/05/2008
 * \version 1.1.0
 *
 * Revision History :
 * ----------  -------  -----------  ----------------------------
 * Date        Version  Author       Description
 * ----------  -------  -----------  ----------------------------
 * 24/03/2008  1.0.0    wiznet		 Release with W5300 launching
 * ----------  -------  -----------  ----------------------------
 *
 * ----------  -------  -----------  ----------------------------
 * http://wiznet.co.kr
 */
 

#include <common.h>
#include <s3c2410.h>
#include <asm/mach-types.h>
#include "lcd.h"

/* ------------------------------------------------------------------------- */

#define FCLK_SPEED 1

#if FCLK_SPEED==0		/* Fout = 203MHz, Fin = 12MHz for Audio */
#define M_MDIV	0xC3
#define M_PDIV	0x4
#define M_SDIV	0x1
#elif FCLK_SPEED==1		/* Fout = 202.8MHz */
#define M_MDIV	0xA1
#define M_PDIV	0x3
#define M_SDIV	0x1
#elif FCLK_SPEED==2		// Fout = 266MHz
#define M_MDIV  0x7d
#define M_PDIV  0x1
#define M_SDIV  0x1
#endif

#define USB_CLOCK 1

#if USB_CLOCK==0
#define U_M_MDIV	0xA1
#define U_M_PDIV	0x3
#define U_M_SDIV	0x1
#elif USB_CLOCK==1
#define U_M_MDIV	0x48
#define U_M_PDIV	0x3
#define U_M_SDIV	0x2
#elif USB_CLOCK==2
#define U_M_MDIV        0x78
#define U_M_PDIV        0x2
#define U_M_SDIV        0x3
#endif



#define GPIO_bit(x)             ( 1 << x )
#define LED_0                   GPIO_bit(10)     // DEBUG LED
#define LED_1                   GPIO_bit(11)     // DEBUG LED
#define __REG(x)                (*((volatile unsigned int *)x))
#define GPGDAT                  __REG(0x56000064) //Port G data


#define led1_on()			GPGDAT &= ~LED_0
#define led1_off()			GPGDAT |= LED_0
#define led1_toggle()		GPGDAT ^= LED_0

#define led2_on()			GPGDAT &= ~LED_1
#define led2_off()			GPGDAT |= LED_1
#define led2_toggle()		GPGDAT ^= LED_1



static inline void delay (unsigned long loops)
{
	__asm__ volatile ("1:\n"
	  "subs %0, %1, #1\n"
	  "bne 1b":"=r" (loops):"0" (loops));
}


void led_display(void)
{
	int i;
	
	led1_off();
	led2_off();

	for (i=0; i<5; i++)
	{
		delay (1000000);
		led1_on();
		led2_off();
		delay (1000000);
		led1_off();		
		led2_on();
	}
	
	led1_off();
	led2_off();
}

void lcd_dsp_logo(void)
{
	char buf[17];

	// 0123456789abcdef	
	sprintf(buf, "W5300E01-ARM V%02d", FIRMWARE_VER);
	
	lcd_gotoxy(0,0);
	lcd_puts(buf);
}



/*
 * Miscellaneous platform dependent initialisations
 */

int board_init (void)
{
	DECLARE_GLOBAL_DATA_PTR;
	S3C24X0_CLOCK_POWER * const clk_power = S3C24X0_GetBase_CLOCK_POWER();
	S3C24X0_GPIO * const gpio = S3C24X0_GetBase_GPIO();
	S3C24X0_MEMCTL * const memctl = S3C24X0_GetBase_MEMCTL();
  
	/* to reduce PLL lock time, adjust the LOCKTIME register */
	clk_power->LOCKTIME = 0xFFFFFF;

	/* configure MPLL */
	clk_power->MPLLCON = ((M_MDIV << 12) + (M_PDIV << 4) + M_SDIV);

	/* some delay between MPLL and UPLL */
	delay (4000);

	/* configure UPLL */
	clk_power->UPLLCON = ((U_M_MDIV << 12) + (U_M_PDIV << 4) + U_M_SDIV);

	/* some delay between MPLL and UPLL */
	delay (8000);

#define CS2_TACS                    ( 0x1 << 13 )   //  1CLK
#define CS2_TCOS                    ( 0x2 << 11 )   //  2CLK
#define CS2_TACC                    ( 0x4 <<  8 )   // 14CLK
#define CS2_TCOH                    ( 0x0 <<  6 )   //  0CLK
#define CS2_TAH                     ( 0x0 <<  4 )   //  0CLK
#define CS2_TACP                    ( 0x0 <<  2 )   //  0CLK
#define CS2_PMC                     ( 0x0 <<  0 )   // normal

	/* set up the memory control */
	memctl->BWSCON = 0x22111110;
	memctl->BANKCON[0] = 0x0700;
	memctl->BANKCON[1] = 0x0700;
	memctl->BANKCON[2] = (CS2_TACS | CS2_TCOS | CS2_TACC | CS2_TCOH | CS2_TAH  | CS2_TACP | CS2_PMC );
	memctl->BANKCON[3] = 0x0700;
	memctl->BANKCON[4] = 0x0700;
	memctl->BANKCON[5] = 0x0700;

	// Trcd = 20ns = 2clock
	memctl->BANKCON[6] = 0x18001;
	memctl->BANKCON[7] = 0x18001;

	// REFEN = 1, TREFMD = 0, Trp = 20ns:2clock:0, Trc = 65ns:7clock:3
	// refresh counter = 489
	memctl->REFRESH = (1<<23) | (3<<18) | 489;

	/* arch number of W5300E01 Board */
	gd->bd->bi_arch_number = MACH_TYPE_W5300E01;

	/* adress of boot parameters */
	gd->bd->bi_boot_params = 0x30000100;

	icache_enable();
	dcache_enable();

	//LCD Initialize	
	lcd_init();
	delay (1000);	
	
	//Logo Display
	lcd_dsp_logo();
	
	led_display();	
	
	
	return 0;
}

int dram_init (void)
{
	DECLARE_GLOBAL_DATA_PTR;

	gd->bd->bi_dram[0].start = PHYS_SDRAM_1;
	gd->bd->bi_dram[0].size = PHYS_SDRAM_1_SIZE;

	return 0;
}

#if (CONFIG_COMMANDS & CFG_CMD_NAND)
extern ulong nand_probe (ulong physadr);

void nand_init (void)
{
	DECLARE_GLOBAL_DATA_PTR;
	S3C2410_NAND * const nand_reg = S3C2410_GetBase_NAND();

	/* set the bus interface characteristics based on
	 * tDS Data Set up Time 30 - ns
	 * tDH Data Hold Time 20 - ns
	 * tALS ALE Set up Time 20 - ns
	 * 16ns at 60 MHz ~= 3
	 */
	/* Setup NAND Flash, fitst enable the address range of GCS2 */
	nand_reg->NFCONF = ((0x1<<15)|(0x0<<12)|(0x1<<11)|(0x0<<8)|(0x3<<4)|(0x0<<0));

#ifdef DEBUG
	printf ("Probing at 0x%.8x\n", S3C2410_NAND_BASE);
#endif

	printf (" %4lu MB\n\n", nand_probe(S3C2410_NAND_BASE) >> 20);
}
#endif

#if (CONFIG_COMMANDS & CFG_CMD_LOADUB)

void ChangeUPllValue(int mdiv,int pdiv,int sdiv)
{
	S3C24X0_CLOCK_POWER * const clk_power = S3C24X0_GetBase_CLOCK_POWER();
	clk_power->UPLLCON = ((mdiv << 12) + (pdiv << 4) + sdiv);
}

void pullup_en(void)
{
	S3C24X0_GPIO * const gpio = S3C24X0_GetBase_GPIO();
	gpio->MISCCR=gpio->MISCCR | (1<<16); // nRSTOUT go to HIGH
}

void pullup_dis(void)
{
	S3C24X0_GPIO * const gpio = S3C24X0_GetBase_GPIO();
	gpio->MISCCR=gpio->MISCCR&~(1<<16); // nRSTOUT go to LOW
}

void usbd_en(void)
{
	S3C24X0_GPIO * const gpio = S3C24X0_GetBase_GPIO();
	gpio->MISCCR=gpio->MISCCR&~(1<<3); // USBD is selected instead of USBH1
  gpio->MISCCR=gpio->MISCCR&~(1<<13); // USB port 1 is enabled.
}

#endif
