/*
*
@file		lcd.c
@brief	LCD Basic Function Implement File

This source file is open, so it's easy to look for through Internet.\n
Do not need to controll LCD.
*/

#include "lcd.h"

#define BUSY		0x80
#define LCD_DELAY	60000

static inline void lcd_delay (unsigned long loops)
{
	__asm__ volatile ("1:\n"
	  "subs %0, %1, #1\n"
	  "bne 1b":"=r" (loops):"0" (loops));
}


/**
@brief	Check to be ready LCD.
@return 	If it's ready ,then '1',else then '-1'. If LCD Device is not insert actually,then return '-1'.
*/
char lcd_ready(void)
{
	unsigned int flag;
	/*
	while( (flag=LcdCommandR) & BUSY)  
	{
		if(flag == 0xFF) return -1;  
	}*/
	
	lcd_delay(LCD_DELAY);
	
	return 1;
}



/**
 @brief		Clear LCD. 
*/
void lcd_clrscr(void)       
{
	lcd_ready();
	LcdCommandW = 0x01;
}



/**
 @brief	Initialize LCD.
 @return 	If initialization is O.K, then return '1', else then return '0'. 
*/
char lcd_init(void)
{
	if((char)-1 ==lcd_ready()) return 0;
	LcdCommandW = 0x38;

	lcd_ready();
	LcdCommandW = 0x0C;

	lcd_clrscr();
	lcd_gotoxy(0,0);
	return 1;
}


/**
 @brief		Move Cursor to X Column, Y Row.
  
 LCD to be offered by WIZnet is '2*16' Dimension, so Row(Argument y) is not above 1.
*/
void lcd_gotoxy(
	unsigned int x, 	/**< x - Column to move(INPUT) */
	unsigned int y		/**< y - Row to move(INPUT) */
	)
{
    lcd_ready();
    switch(y)
    {
        case 0 : LcdCommandW = 0x80 + x; break;
        case 1 : LcdCommandW = 0xC0 + x; break;
        case 2 : LcdCommandW = 0x94 + x; break;
        case 3 : LcdCommandW = 0xD4 + x; break;
    }
}


/**
 @brief		Output character string in current Cursor.
 @return 	Character string's Pointer to be output 
*/
char * lcd_puts(
	char* str		/**< str - Character to be output in LCD. (INPUT) */
	)
{
	unsigned char i;

	for (i=0; str[i] != '\0'; i++){
		lcd_ready();
		LcdDataW = str[i];
	}
	return str;
}


/**
 @brief		Output 1 character in current Cursor.
 @return 	Character string's Pointer to be output 
*/
void lcd_putch(
	char ch	/**< str - Character to be output in LCD. (INPUT) */
	)
{
	lcd_ready();
	LcdDataW = ch;
}


#ifndef __LCD_UNUSED
/**
 @brief		Decide Cursor type.
*/
void lcd_set_cursor_type(
	unsigned int type	/**< type - Cursor type(INPUT) */
	)	
{
	lcd_ready();
	switch(type) 
	{
			//No Cursor 
	        case 0 : LcdCommandW = 0x0C; break;  
			// Normal Cursor 
	        case 1 : LcdCommandW = 0x0E; break; 
		// No Cursor | Blink
		case 2 : LcdCommandW = 0x0D; break; 
			// Normal Cursor | Blink 
	        case 3 : LcdCommandW = 0x0F; break; 
}


/**
 @brief		Shift to Left and Right current Cursor.
*/
void ShiftCursor(
	unsigned int dir	/**< dir - Decide direction to be Shift.(INPUT)  dir !=0  -> Right Shift, dir == 0 -> Left Shift */
	)	
{
	lcd_ready();
	LcdCommandW = (dir ? 0x14: 0x10);
}


/**
 @brief		Move Cursor first Column.
*/
void lcd_home_cursor(void)       
{
	lcd_ready();
	LcdCommandW = 2;
}

#endif
